/**********************************************************************
*	Bastion Products Copyright 1998								*
*	All rights reserved										*
*	This source is freely distributable in an unmodified state.		*
*	Project:			Blithe Static Library						*
*	Source:			Blithe.c									*
*	Last Modified:	5/20/98									*
*	Author:			Jennifer Weston							*
*	Description:		This project creates a static library which	*
*					provides weak linking to the Blithe shared	*
*					library and a C++ wrapper class. The Blithe.c	*
*					file provides the C weak linking.			*
***********************************************************************/
#include "Blithe.h"			// C prototypes
#include "BlitheGlobal.h"		// Global function variables
#include <image.h>
#include <Resources.h>
#include <Path.h>
#include <FindDirectory.h>
#include <Directory.h>
#include <cstring>
#include <cstdlib>
#include <Roster.h>
#include <Application.h>
#include <cstdio>
#include <Errors.h>

#define BLITHE_LIB "blithe.so"	// File name of the shared library

// Function prototypes used only within this file
image_id hasSharedLib(void);		// Find the Blithe shared library
BResources *FindImageDictionary(void);	// Finds the image dictionary
BResources *FindAppDictionary(void);		// Finds the internal application dictionary
char *GetHex(int32 inStringID);		// Gets a hexadecimal string of an int32
int32 GetAppID(void);				// Finds the ID of the language built into the application
void ConvertEscapes(char* entry);	// Convert escape codes to their proper form

/**********************************************************************
*	Function:		blithe_init									*
*	Arguments:	<none>										*
*	Returns:		status_t	-	0 for sucess						*
*	Description:	Links with the shared library if it's present;		*
*				otherwise, uses it's internal calls.				*
**********************************************************************/
status_t blithe_init(void)
{
/* If we find the lib, set up the calls */
	image_id the_image;
	the_image = hasSharedLib();
	if (the_image != B_ERROR)
	{
		status_t err;
		err = get_image_symbol(the_image, "has_application_dictionary", B_SYMBOL_TYPE_TEXT, (void**)&has_application_dictionary);
		if (err) goto END_BLITHE_INIT;
		err = get_image_symbol(the_image, "get_default_language", B_SYMBOL_TYPE_TEXT, (void**)&get_default_language);
		if (err) goto END_BLITHE_INIT;
		err = get_image_symbol(the_image, "get_default_language", B_SYMBOL_TYPE_TEXT, (void**)&get_default_language);
		if (err) goto END_BLITHE_INIT;
		err = get_image_symbol(the_image, "get_entry", B_SYMBOL_TYPE_TEXT, (void**)&get_entry);
		if (err) goto END_BLITHE_INIT;
		err = get_image_symbol(the_image, "get_blob", B_SYMBOL_TYPE_TEXT, (void**)&get_blob);
		if (err) goto END_BLITHE_INIT;
		err = get_image_symbol(the_image, "get_entry_length", B_SYMBOL_TYPE_TEXT, (void**)&get_entry_length);
		if (err) goto END_BLITHE_INIT;
		err = get_image_symbol(the_image, "get_max_length",B_SYMBOL_TYPE_TEXT, (void**)&get_max_length);
		if (err) goto END_BLITHE_INIT;
		err = get_image_symbol(the_image, "language_name", B_SYMBOL_TYPE_TEXT, (void**)&language_name);
		if (err) goto END_BLITHE_INIT;
		err = get_image_symbol(the_image, "language_id", B_SYMBOL_TYPE_TEXT, (void**)&language_id);
		if (err) goto END_BLITHE_INIT;
		err = get_image_symbol(the_image, "get_raw_bits", B_SYMBOL_TYPE_TEXT, (void**)&get_raw_bits);

END_BLITHE_INIT:
		if (!err) return B_OK;
	}

/* Otherwise, use the static ones */
	has_application_dictionary = static_has_application_dictionary;
	get_default_language = static_get_default_language;
	get_entry = static_get_entry;
	get_entry_length = static_get_entry_length;
	get_blob = static_get_blob;
	get_max_length = static_get_max_length;
	get_language_list = static_get_language_list;
	language_name = static_language_name;
	language_id = static_language_id;
	get_raw_bits = static_get_raw_bits;
	return B_OK;
}

/**********************************************************************
*	Function:		hasSharedLib									*
*	Arguments:	<none>										*
*	Returns:		image_id	-	the image id if found, else B_ERROR	*
*	Description:	Searches the BeOS, common, and user lib 			*
*				directories for the shared library and returns the	*
*				image id. If there is a problem, it returns B_ERROR	*
**********************************************************************/
image_id hasSharedLib(void)
{
/* Look for the lib in the BeOS lib directory*/
	status_t err;
	BPath thePath;
	image_id the_image;

	err = find_directory(B_BEOS_LIB_DIRECTORY,&thePath);
	if (err == B_OK)
	{
		thePath.Append(BLITHE_LIB);
		the_image = load_add_on(thePath.Path());
		if (the_image > 0) return the_image;
	}

/* Look for the lib in the Common lib directory */
	err = find_directory(B_COMMON_LIB_DIRECTORY,&thePath);
	if (err == B_OK)
	{
		thePath.Append(BLITHE_LIB);
		BDirectory commonTest(thePath.Path());
		the_image = load_add_on(thePath.Path());
		if (the_image > 0) return the_image;
	}

/* Look for the lib in the user lib directory */
	err = find_directory(B_USER_LIB_DIRECTORY,&thePath);
	if (err == B_OK)
	{
		thePath.Append(BLITHE_LIB);
		BDirectory userTest(thePath.Path());
		the_image = load_add_on(thePath.Path());
		if (the_image > 0) return the_image;
	}
	return B_ERROR;
}

/**********************************************************************
*	Function:		static_has_application_dictionary				*
*	Arguments:	int32	inLangID								*
							language ID						*
*	Returns:		int		returns 1 if inLangID is the language		*
*						ID built into the applicaton			*
*	Description:	Checks the language ID internal to the application	*
*				and returns 1 if they are the same and 0 if they 	*
*				are not. This function is used for 				*
*				has_application_dictionary if the shared library	*
*				is not found.									*
**********************************************************************/
int static_has_application_dictionary(int32 inLangID)
{
	return (GetAppID() == inLangID) ? 1 : 0;
}

/**********************************************************************
*	Function:		static_get_default_language						*
*	Arguments:	<none>										*
*	Returns:		int32	returns the language ID built into the 	*
*						applicaton							*
*	Description:	Finds the language ID built into the application.	*
*				This function is used for 						*
*				get_default_language if the shared library is not	*
*				found.										*
**********************************************************************/
int32 static_get_default_language(void)
{
	return GetAppID();
}

/**********************************************************************
*	Function:		static_get_entry								*
*	Arguments:	int32	inStringID							*
*							the string ID that is requested		*
*				int32	inLangID								*
*							the language ID that is requested	*
*				char*	inBuffer								*
*							an optional buffer, NULL if unused	*
*				int32	inBuffSize							*
*							the length of inBuffer				*
*	Returns:		char*	returns a buffer for a string			*
*	Description:	Returns the string associated with inStringID.	This	*
*				function only looks in the built in application	*
*				strings, so the inLangID is ignored. If inBuffer is	*
*				provided, it will copy at most inBuffSize 		*
*				characters of the string into inBuffer and return	*
*				inBuffer. This function is used for get_entry if 	*
*				the shared library	is not found.					*
**********************************************************************/
char* static_get_entry(int32 inStringID, int32 /*inLangID*/, char* inBuffer, int32 inBuffSize)
{
// Get it from this image
	BResources *theRes = FindImageDictionary();
	char *entry=NULL;
	size_t len=0;
	if (theRes) entry = (char*)(theRes->FindResource('CSTR',inStringID,&len));
	delete theRes;
	if (!entry)
	{
		theRes = FindAppDictionary();
		if (theRes) entry = (char*)(theRes->FindResource('CSTR',inStringID,&len));
		delete theRes;
	}

// Failsafe
	if (!entry) return GetHex(inStringID);

// Transfer it to inBuffer if it exists
	if (inBuffer && inBuffSize)
	{
		strncpy(inBuffer,entry,inBuffSize-1);
		inBuffer[inBuffSize] = 0;
		free(entry);
		entry = inBuffer;
	}

// Convert escape characters to their proper codes
	ConvertEscapes(entry);

	return entry;
}

/**********************************************************************
*	Function:		static_get_blob								*
*	Arguments:	int32	inResType							*
*							the requested resource type		*
*				int32	inResID							*
*							the requested resource ID		*
*				int32	inLangID								*
*							the language ID that is requested	*
*				void*	inBuffer								*
*							an optional buffer for the data	*
*				size_t	inBuffLength								*
*							the length of inBuffer			*
*				int32*	outBlobSize							*
*							stores the size of the data		*
*							contained by the returned pointer	*
*	Returns:		void*	returns a pointer to the requested data			*
*	Description:	Returns the data stored in the requested		*
*				resource.	This	function only looks in the built	*
*				in application resources, so the inLangID is		*
*				ignored. This function is used for get_blob if 	*
*				the shared library	is not found.					*
**********************************************************************/
void* static_get_blob(uint32 inResType,int32 inResID, int32 /*inLangID*/,void* inBuffer,size_t inBuffLength, size_t* outBlobSize)
{
// Get it from this image
	BResources *theRes = FindImageDictionary();
	void *blob=NULL;
	if (theRes) blob = theRes->FindResource((type_code)inResType,inResID,outBlobSize);
	delete theRes;
	if (!blob)
	{
		theRes = FindAppDictionary();
		if (theRes) blob = theRes->FindResource((type_code)inResType,inResID,outBlobSize);
		delete theRes;
	}

// Copy the data into a persistent buffer
	if (blob)
	{
	// Create a buffer if necessary
		if (inBuffLength < 1 || inBuffer == NULL)
		{
			inBuffLength = *outBlobSize;
			inBuffer = malloc(inBuffLength);
		}
		memcpy(inBuffer,blob,inBuffLength);
	}

// No failsafe since it's up to the user to make sure that these types/ids exist
	return blob;
}

/**********************************************************************
*	Function:		static_get_entry_length							*
*	Arguments:	int32	inStringID							*
*							the string ID that is requested		*
*				int32	inLangID								*
*							the language ID that is requested	*
*	Returns:		int32	returns the number of bytes in the 		*
*						requested string						*
*	Description:	This	function only looks in the built in			* 
*				application strings, so the inLangID is ignored.	*
*				This function is used for get_entry_length if the	*
*				shared library is not found.					*
**********************************************************************/
int32 static_get_entry_length(int32 inStringID, int32 /*inLangID*/)
{
// Get it from the app
	BResources *theRes = FindImageDictionary();
	char *entry = NULL;
	size_t len=0;
	if (theRes) entry = (char*)(theRes->FindResource('CSTR',inStringID,&len));
	delete theRes;

	if (!entry)
	{
		theRes = FindAppDictionary();
		if (theRes) entry = (char*)(theRes->FindResource('CSTR',inStringID,&len));
		delete theRes;
	}

	if (entry)
	{
		len = strlen(entry);
		free(entry);
	}

	return len;
}

/**********************************************************************
*	Function:		static_get_max_length							*
*	Arguments:	int32	inLangID								*
*							the language ID that is requested	*
*	Returns:		int32	returns the maximum number of bytes for	*
*						inLangID								*
*	Description:	Checks the built-in dictionary for max length		*
**********************************************************************/
int32 static_get_max_length(int32 /*inLangID*/)
{
// Get it from the image
	int32 *length=NULL;
	size_t len = 0;
	BResources *theRes = FindImageDictionary();
	if (theRes) length = (int32*)(theRes->FindResource('LONG',2,&len));
	delete theRes;

	if (!length)
	{
		theRes = FindAppDictionary();
		if (theRes) length = (int32*)(theRes->FindResource('LONG',2,&len));
		delete theRes;
	}

	int32 maxLength=0;
	if (length)
	{
		maxLength = *length;
		free(length);
	}
	return maxLength;
}

/**********************************************************************
*	Function:		static_get_language_list						*
*	Arguments:	char**	outList								*
*							the address of the array of strings	*
*							that is returned.					*
*	Returns:		int32	returns 1								*
*	Description:	This	function returns the name of the language		* 
*				built into the application in outList. The memory	*
*				for the name is allocated as a single char* and	*
*				a buffer of char. This function is used for		*
*				get_language_list if the shared library is not		*
*				found.										*
**********************************************************************/
int32 static_get_language_list(char** outList)
{
// Get it from the image
	*outList = (char*)malloc(sizeof(char*));
	int32 lang=0;
	BResources *theRes = FindImageDictionary();
	size_t len;
	int32 i=0;
	char *c = NULL;
	if (theRes) c = (char*)(theRes->FindResource('CSTR',(int32)0,&len));
	delete theRes;

	if (!c)
	{
		theRes = FindAppDictionary();
		if (theRes) c = (char*)(theRes->FindResource('CSTR',(int32)0,&len));
		delete theRes;
		if (!c) return 0;
	}

// Check the list
	while (strcmp(outList[i],c) != 0 && i<=lang) ++i;
	if (i > lang)
	{
		outList[i] = c;
		lang = i;
	}
	return lang;
}

/**********************************************************************
*	Function:		static_language_name							*
*	Arguments:	int32	inLangID								*
*							the requested language ID			*
*				char*	inBuffer								*
*							an optional buffer for the name		*
*				int32	inBuffSize							*
*							the size of inBuffer				*
*	Returns:		int32	returns the buffer with the name			*
*	Description:	This	function returns the name of the language		* 
*				built into the application in outList. The memory	*
*				for the name is allocated as a single char* and	*
*				a buffer of char. This function is used for		*
*				language_name if the shared library is not found.	*
**********************************************************************/
char* static_language_name(int32 /*inLangID*/, char* inBuffer, int32 inBuffSize)
{
	BResources *theRes = FindImageDictionary();
	size_t len;
	char *theName = NULL;
	if (theRes) theName = (char*)(theRes->FindResource('CSTR',(int32)0,&len));
	delete theRes;
	if (!theName)
	{
		theRes = FindAppDictionary();
		if (theRes) theName = (char*)(theRes->FindResource('CSTR',(int32)0,&len));
		delete theRes;
	}

	if (!theName) return NULL;
	if (!inBuffer) return theName;
	strncpy(inBuffer,theName,inBuffSize-1);
	inBuffer[inBuffSize] = (char)0;
	return inBuffer;
}

/**********************************************************************
*	Function:		static_language_id								*
*	Arguments:	char*	inName								*
*							the requested language name			*
*	Returns:		int32	returns the language ID of the application*
*	Description:	This	function returns the ID of the language built	*
*				into the application in outList. The memory for the	*
*				name is allocated as a single char* and a buffer	*
*				of char. This function is used for language_id if	*
*				the shared library is not found.					*
**********************************************************************/
int32 static_language_id(char */*inName*/)
{
	return GetAppID();
}

/**********************************************************************
*	Function:		static_get_raw_bits							*
*	Arguments:	<none>										*
*	Returns:		void*	NULL									*
*	Description:	This function is used for get_raw_bits if	the		*
*				shared library is not found. It returns NULL.		*
**********************************************************************/
size_t static_get_raw_bits(void** theBits)
{
	*theBits = NULL;
	return 0;
}

/**********************************************************************
*	Function:		FindImageDictionary							*
*	Arguments:	<none>										*
*	Returns:		BResources*	returns the resource of the app		*
*	Description:	This	function returns the resource of the app. 	*
**********************************************************************/
BResources *FindImageDictionary(void)
{
	BFile file; 
	int32 cookie=0;
	image_info info;
	int32 myPtr = (int32)static_get_entry;

	while (get_next_image_info(0, &cookie, &info) == B_OK && 
			!(myPtr < (int32)info.data && myPtr > (int32)info.data+info.data_size)){}
	file.SetTo(info.name, B_READ_ONLY);
	return new BResources(&file);
}

/**********************************************************************
*	Function:		FindAppDictionary								*
*	Arguments:	<none>										*
*	Returns:		BResources*	returns the resource of the app		*
*	Description:	This	function returns the resource of the app. 	*
**********************************************************************/
BResources *FindAppDictionary(void)
{
	BFile file; 
	app_info info;

	be_app->GetAppInfo(&info);
	file.SetTo(&info.ref, B_READ_ONLY);
	return new BResources(&file);
}


/**********************************************************************
*	Function:		GetHex										*
*	Arguments:	int32	inStringID							*
*							the requested string ID				*
*	Returns:		char*	the string which is the hex of inStringID	*
*	Description:	This	function converts the inStringID to a string	*
*				in hex										*
**********************************************************************/
char *GetHex(int32 inStringID)
{
	char *theString = (char*)(malloc(11));
	sprintf(theString,"%lx",inStringID);
	return theString;
}

/**********************************************************************
*	Function:		GetAppID										*
*	Arguments:	<none>										*
*	Returns:		int32	the language ID of the language built		*
*						into the application.					*
*	Description:	This	function reads and returns the language ID	*
*				found in the application.						*
**********************************************************************/
int32 GetAppID(void)
{
	BResources *theRes = FindImageDictionary();
	size_t len;
	int32 *id = NULL;
	if (theRes) id = (int32*)(theRes->FindResource('LONG',1,&len));
	delete theRes;
	if (!id)
	{
		theRes = FindAppDictionary();
		if (theRes) id = (int32*)(theRes->FindResource('LONG',1,&len));
		delete theRes;
	}

	if (!id) return 0;
	int lang = *id;
	free(id);
	return lang;
}

/**********************************************************************
*	Function:		GetAppID										*
*	Arguments:	char*	entry								*
*							the entry string					*
*	Returns:		<none>										*
*	Description:	Converts escape codes to their proper form.		*
**********************************************************************/
void ConvertEscapes(char* entry)
{
// Iterate over each character
	char* t=entry;
	char* temp = (char*)(malloc(strlen(entry)+1));
	while (t && t[0])
	{
	// Is t a '\'
		if (t[0] == '\\')
		{
		// Convert it to the proper code based on t[1]
			switch (t[1])
			{
			case 'n':
			// New line (in BeOS linefeed)
				t[0] = B_ENTER;
				break;
			case 'b':
			// Backspace
				t[0] = B_BACKSPACE;
				break;
			case 'f':
			// Formfeed
				t[0] = B_PAGE_DOWN;
				break;
			case 'r':
			// Carriage return
				t[0] = 13;
				break;
			case 't':
			// Horizontal tab
				t[0] = B_TAB;
				break;
			case '\"':
			// Double quote
				--t;
				break;
			case '\'':
			// Single quote
				--t;
				break;
			case '0':
			// Null
				t[0] = 0;
				break;
			case '\\':
			// Backslash
				--t;
				break;
			}

		// Move the string over
			strcpy(temp,t+2);
			strcpy(t+1,temp);
		}
		++t;
	}
}